// Ubicación: public_html/asesoria.virtualex.ec/js/app.js

document.addEventListener('DOMContentLoaded', () => {
    const elements = {
        form: document.getElementById('chatForm'),
        input: document.getElementById('userQuery'),
        history: document.getElementById('chatHistory'),
        citations: document.getElementById('citationsList'),
        btn: document.getElementById('sendBtn'),
        balance: document.getElementById('balanceDisplay'),
        docsCount: document.getElementById('docsCount')
    };

    // Auto-focus al iniciar
    elements.input.focus();

    elements.form.addEventListener('submit', async (e) => {
        e.preventDefault();
        const query = elements.input.value.trim();
        if (!query) return;

        // 1. UI: Bloquear input y mostrar pregunta
        appendMessage('user', query);
        elements.input.value = '';
        toggleInput(false);

        // 2. UI: Mostrar Loading
        const loadingId = appendLoading();

        try {
            // Llamada al endpoint definido en index.php
            const response = await fetch('/api/chat', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ query: query })
            });

            // Manejo de errores HTTP
            if (!response.ok) throw new Error(`Error HTTP: ${response.status}`);

            const data = await response.json();
            removeLoading(loadingId);

            if (data.error) {
                appendMessage('error', `Error: ${data.error}`);
            } else {
                appendMessage('ai', data.answer);
                updateCitations(data.citations);
                
                // Actualizar saldo si viene en la respuesta
                if (data.balance !== undefined) {
                    elements.balance.innerText = data.balance;
                    // Efecto visual de actualización
                    elements.balance.classList.add('text-green-500');
                    setTimeout(() => elements.balance.classList.remove('text-green-500'), 1000);
                }
            }

        } catch (err) {
            removeLoading(loadingId);
            appendMessage('error', 'Error de conexión con el servidor. Intenta nuevamente.');
            console.error(err);
        } finally {
            toggleInput(true);
            elements.input.focus();
        }
    });

    // Helper: Habilitar/Deshabilitar formulario
    function toggleInput(enabled) {
        elements.input.disabled = !enabled;
        elements.btn.disabled = !enabled;
        elements.btn.classList.toggle('opacity-50', !enabled);
    }

    // Helper: Agregar mensaje al chat
    function appendMessage(role, text) {
        const div = document.createElement('div');
        
        if (role === 'user') {
            div.className = 'ml-auto bg-blue-600 text-white p-4 rounded-xl rounded-tr-none max-w-[85%] shadow-sm';
            div.textContent = text;
        } else if (role === 'ai') {
            div.className = 'mr-auto bg-white border border-gray-200 text-gray-800 p-5 rounded-xl rounded-tl-none max-w-[90%] shadow-sm prose prose-sm prose-blue';
            div.innerHTML = marked.parse(text);
        } else if (role === 'error') {
            div.className = 'mx-auto bg-red-50 border border-red-100 text-red-600 p-3 rounded-lg text-center text-sm w-full';
            div.textContent = text;
        }

        elements.history.appendChild(div);
        scrollToBottom();
    }

    // Helper: Animación de carga
    function appendLoading() {
        const id = 'load-' + Date.now();
        const div = document.createElement('div');
        div.id = id;
        div.className = 'mr-auto bg-white border border-gray-100 p-4 rounded-xl rounded-tl-none shadow-sm flex items-center gap-2';
        div.innerHTML = `
            <div class="flex space-x-1">
                <div class="w-2 h-2 bg-blue-400 rounded-full animate-bounce"></div>
                <div class="w-2 h-2 bg-blue-400 rounded-full animate-bounce delay-75"></div>
                <div class="w-2 h-2 bg-blue-400 rounded-full animate-bounce delay-150"></div>
            </div>
            <span class="text-xs text-gray-400 ml-2">Analizando leyes...</span>
        `;
        elements.history.appendChild(div);
        scrollToBottom();
        return id;
    }

    function removeLoading(id) {
        const el = document.getElementById(id);
        if(el) el.remove();
    }

    function scrollToBottom() {
        elements.history.scrollTop = elements.history.scrollHeight;
    }

    // Helper: Actualizar lista lateral de fuentes
    function updateCitations(docs) {
        elements.citations.innerHTML = '';
        
        if (!docs || docs.length === 0) {
            elements.docsCount.innerText = '0 docs';
            elements.citations.innerHTML = '<div class="text-center text-xs text-gray-400 mt-4">No se citaron fuentes específicas.</div>';
            return;
        }

        elements.docsCount.innerText = `${docs.length} docs`;
        
        docs.forEach(doc => {
            const item = document.createElement('div');
            item.className = 'p-3 bg-white border border-gray-200 rounded-lg hover:border-blue-300 hover:shadow-md transition-all cursor-pointer group';
            item.innerHTML = `
                <div class="flex items-start gap-3">
                    <div class="bg-blue-50 text-blue-600 p-1.5 rounded group-hover:bg-blue-100 transition-colors">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path></svg>
                    </div>
                    <div class="overflow-hidden">
                        <h4 class="font-bold text-gray-700 text-xs truncate group-hover:text-blue-700" title="${doc.titulo}">${doc.titulo}</h4>
                        <div class="flex items-center gap-2 mt-1">
                            <span class="text-[10px] text-gray-400 bg-gray-100 px-1.5 py-0.5 rounded">ID: ${doc.id}</span>
                            <span class="text-[10px] text-gray-400">📅 ${doc.fecha}</span>
                        </div>
                    </div>
                </div>
            `;
            elements.citations.appendChild(item);
        });
    }
});