<?php
// Ubicación: public_html/asesoria.virtualex.ec/app/Services/LegacySearchService.php

namespace App\Services;

use App\Config\Database;
use PDO;

class LegacySearchService {
    
    public function search(string $query, int $limit = 5): array {
        $db = Database::getLegacy();
        
        // Consulta optimizada FULLTEXT para MyISAM
        $sql = "SELECT
                    id_documento,
                    titulo,
                    MATCH(titulo, contenido) AGAINST (:query IN NATURAL LANGUAGE MODE) AS score,
                    contenido,
                    archivotxt,
                    resumen,
                    emisor,
                    fecha1
                FROM documento
                WHERE
                    estado = 1
                    AND vigencia = 1
                    AND desplegado = 1
                    AND MATCH(titulo, contenido) AGAINST (:query IN NATURAL LANGUAGE MODE)
                ORDER BY score DESC
                LIMIT :limit";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':query', $query, PDO::PARAM_STR);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        $results = $stmt->fetchAll();
        $processed = [];

        foreach ($results as $row) {
            // Prioridad de contenido: campo 'contenido' > 'archivotxt'
            $rawText = !empty($row['contenido']) ? $row['contenido'] : $row['archivotxt'];
            
            // --- CONVERSIÓN CRÍTICA: Latin1 a UTF-8 ---
            // La IA (Gemini) y JSON requieren UTF-8. La base viene en ISO-8859-1.
            $tituloUtf8 = mb_convert_encoding($row['titulo'], "UTF-8", "ISO-8859-1");
            $contenidoUtf8 = mb_convert_encoding($rawText, "UTF-8", "ISO-8859-1");
            $resumenUtf8 = mb_convert_encoding($row['resumen'] ?? '', "UTF-8", "ISO-8859-1");
            $emisorUtf8 = mb_convert_encoding($row['emisor'] ?? '', "UTF-8", "ISO-8859-1");

            // Limpieza básica de HTML y recorte para ventana de contexto
            $cleanContent = strip_tags($contenidoUtf8);
            // Limitamos a ~12000 caracteres (aprox 3000 tokens) por documento
            $cleanContent = substr($cleanContent, 0, 12000); 

            $processed[] = [
                'id' => $row['id_documento'],
                'titulo' => $tituloUtf8,
                'fecha' => $row['fecha1'],
                'emisor' => $emisorUtf8,
                'contenido' => $cleanContent,
                'resumen' => $resumenUtf8,
                'score' => $row['score']
            ];
        }

        return $processed;
    }
}