<?php
// Ubicación: public_html/asesoria.virtualex.ec/app/Controllers/ChatController.php

namespace App\Controllers;

use App\Config\Database;
use App\Services\LegacySearchService;
use App\Services\GeminiService;
use PDO;
use Exception;

class ChatController {
    
    public function handleRequest() {
        session_start();
        
        // Simulación de sesión (Para desarrollo, ID 1 fijo. En prod, borrar esto)
        if (!isset($_SESSION['user_id'])) {
             $_SESSION['user_id'] = 1; 
        }

        $userId = $_SESSION['user_id'];
        
        // Leer input JSON
        $input = json_decode(file_get_contents('php://input'), true);
        $query = trim($input['query'] ?? '');

        if (empty($query)) {
            echo json_encode(['error' => 'La consulta no puede estar vacía']);
            return;
        }

        try {
            $dbApp = Database::getApp();

            // 1. Validar Créditos
            $stmt = $dbApp->prepare("SELECT balance FROM credit_balance WHERE user_id = ?");
            $stmt->execute([$userId]);
            $balance = $stmt->fetchColumn();

            // Validar si existe usuario o saldo
            if ($balance === false) {
                 // Auto-crear saldo para el user 1 de prueba si no existe
                 $dbApp->exec("INSERT IGNORE INTO credit_balance (user_id, balance) VALUES (1, 10)");
                 $balance = 10;
            }

            if ($balance < 1) {
                echo json_encode(['error' => 'Saldo insuficiente.']);
                return;
            }

            // 2. Búsqueda RAG (Legacy DB)
            $searchService = new LegacySearchService();
            $documents = $searchService->search($query, 5);

            if (empty($documents)) {
                echo json_encode([
                    'answer' => 'No se encontraron documentos relevantes en la base de datos para responder tu consulta.',
                    'citations' => [],
                    'credits_remaining' => $balance
                ]);
                return;
            }

            // 3. Generación IA (Gemini)
            $gemini = new GeminiService();
            $aiResult = $gemini->generateResponse($query, $documents);

            // 4. Transacción (Descuento y Logs)
            $this->processTransaction($dbApp, $userId, $query, $aiResult, $documents);

            // 5. Respuesta JSON
            echo json_encode([
                'status' => 'success',
                'answer' => $aiResult['text'],
                'citations' => array_map(fn($d) => [
                    'id' => $d['id'], 
                    'titulo' => $d['titulo'], 
                    'fecha' => $d['fecha']
                ], $documents),
                'balance' => $balance - 1
            ]);

        } catch (Exception $e) {
            error_log("ChatController Error: " . $e->getMessage());
            echo json_encode(['error' => 'Error interno procesando la solicitud.']);
        }
    }

    private function processTransaction(PDO $db, int $userId, string $query, array $aiResult, array $documents) {
        $db->beginTransaction();
        try {
            // A. Descontar crédito
            $stmtDed = $db->prepare("UPDATE credit_balance SET balance = balance - 1 WHERE user_id = ?");
            $stmtDed->execute([$userId]);

            // B. Registrar Transacción
            $stmtTrans = $db->prepare("INSERT INTO credit_transactions (user_id, type, amount, description) VALUES (?, 'CONSUMPTION', -1, 'Consulta IA')");
            $stmtTrans->execute([$userId]);

            // C. Guardar Consulta
            $stmtCons = $db->prepare("INSERT INTO consultas (user_id, pregunta, respuesta_ia, tokens_total) VALUES (?, ?, ?, ?)");
            $stmtCons->execute([$userId, $query, $aiResult['text'], ($aiResult['input_tokens'] + $aiResult['output_tokens'])]);
            $consultaId = $db->lastInsertId();

            // D. Guardar Documentos Citados
            $stmtDoc = $db->prepare("INSERT INTO consulta_documentos (consulta_id, doc_id_legacy, titulo, score_relevancia) VALUES (?, ?, ?, ?)");
            foreach ($documents as $doc) {
                $stmtDoc->execute([$consultaId, $doc['id'], $doc['titulo'], $doc['score']]);
            }
            
            // E. Log Técnico
            $stmtLog = $db->prepare("INSERT INTO logs_api (user_id, tokens_input, tokens_output, latency_ms, status_code) VALUES (?, ?, ?, ?, ?)");
            $stmtLog->execute([$userId, $aiResult['input_tokens'], $aiResult['output_tokens'], $aiResult['latency'], $aiResult['status']]);

            $db->commit();
        } catch (Exception $e) {
            $db->rollBack();
            throw $e; // Re-lanzar para que el controller maneje el error
        }
    }
}